﻿// --------------------------------------------------------------------------------------------------------------------
// <copyright file="DateTimeWindow.xaml.cs" company="Palizafzar">
//   Author : Mohammad Habibzadeh
// </copyright>
// <summary>
//   Device Date/Time Setting Window
// </summary>
// --------------------------------------------------------------------------------------------------------------------

namespace PalizTiara.Server
{
    using System;
    using System.Collections.ObjectModel;
    using System.Windows;
    using Api;
    using Api.CallBacks;
    using Api.Definition;
    using Api.Helpers;
    using Api.Models;
    using Protocol.Utility;
    using ViewModel;

    /// <summary>
    /// Device Date/Time Setting Window
    /// </summary>
    public partial class DateTimeWindow
    {
        private readonly TiaraServerManager serverManager;
        private readonly ObservableCollection<DeviceListBoxModel> deviceList = new ObservableCollection<DeviceListBoxModel>();

        public DateTimeWindow(TiaraServerManager serverManager)
        {
            this.InitializeComponent();
            this.serverManager = serverManager;
            this.DeviceListBox.ItemsSource = this.deviceList;
        }

        protected override void OnClosed(EventArgs e)
        {
            this.serverManager.SetTimeEvent -= this.ServerManagerOnSetTimeEvent;
            this.serverManager.SetDateEvent -= this.ServerManagerOnSetTimeEvent;
            this.serverManager.SetDateTimeEvent -= this.ServerManagerOnSetTimeEvent;
            this.serverManager.SetClockTypeEvent -= this.ServerManagerOnSetTimeEvent;
            this.serverManager.SetDateTypeEvent -= this.ServerManagerOnSetTimeEvent;

            base.OnClosed(e);
        }

        private async void DeviceListBox_OnLoaded(object sender, RoutedEventArgs e)
        {
            var devices = await this.serverManager.GetStatusAsyncTask();
            foreach (var device in devices)
            {
                this.deviceList.Add(new DeviceListBoxModel(device.TerminalName, false));
            }
            this.serverManager.SetTimeEvent += this.ServerManagerOnSetTimeEvent;
            this.serverManager.SetDateEvent += this.ServerManagerOnSetTimeEvent;
            this.serverManager.SetDateTimeEvent += this.ServerManagerOnSetTimeEvent;
            this.serverManager.SetClockTypeEvent += this.ServerManagerOnSetTimeEvent;
            this.serverManager.SetDateTypeEvent += this.ServerManagerOnSetTimeEvent;
        }

        private void ServerManagerOnSetTimeEvent(object sender, SetActionEventArgs args)
        {
            Trace.WriteLine(
                TraceLevel.Verbose,
                args.Result ? $"Set Time/Date Done For {args.TerminalName}" : $"Set Time/Date Failed For {args.TerminalName}");
        }

        private async void SetTimeButton_OnClick(object sender, RoutedEventArgs e)
        {
            try
            {
                int hour = int.Parse(this.HourTextBox.Text);
                int min = int.Parse(this.MinuteTextBox.Text);
                int sec = int.Parse(this.SecondTextBox.Text);
                var time = new TimeModel(hour, min, sec);
                foreach (var device in this.deviceList)
                {
                    if (device.IsChecked)
                    {
                        await this.serverManager.SetTimeAsyncTask(time, device.Name);
                    }
                }
            }
            catch (Exception ex)
            {
                Trace.WriteLine(TraceLevel.Error, ex.Message);
            }
        }

        private async void SetDateButton_OnClick(object sender, RoutedEventArgs e)
        {
            var date = this.SetDatePicker.SelectedDate;
            if (date == null)
            {
                return;
            }
            var epochTime = StaticHelpers.GetEpochTime((DateTime)date);
            var dateData = new DateModel(epochTime);
            foreach (var device in this.deviceList)
            {
                if (device.IsChecked)
                {
                    await this.serverManager.SetDateAsyncTask(dateData, device.Name);
                }
            }
        }

        private async void SetDateTimeButton_OnClick(object sender, RoutedEventArgs e)
        {
            var date = this.SetTimeDatePicker.SelectedDate;
            if (date == null)
            {
                return;
            }
            try
            {
                int hour = int.Parse(this.DateHourTextBox.Text);
                int min = int.Parse(this.DateMinuteTextBox.Text);
                int sec = int.Parse(this.DateSecondTextBox.Text);
                date = date?.AddHours(hour);
                date = date?.AddMinutes(min);
                date = date?.AddSeconds(sec);
                var epochTime = StaticHelpers.GetEpochTime((DateTime)date);
                var dateData = new DateModel(epochTime);
                foreach (var device in this.deviceList)
                {
                    if (device.IsChecked)
                    {
                        await this.serverManager.SetDateTimeAsyncTask(dateData, device.Name);
                    }
                }
            }
            catch (Exception ex)
            {
                Trace.WriteLine(TraceLevel.Error, ex.Message);
            }
        }

        private async void SetClockTypeButton_OnClick(object sender, RoutedEventArgs e)
        {
            var type = new ClockTypeModel();
            if (this.DigitalClockRadio.IsChecked.HasValue && this.DigitalClockRadio.IsChecked.Value)
            {
                type.SetClockType(ClockTypes.DIGITAL);
            }
            else if (this.AnalogClockRadio.IsChecked.HasValue && this.AnalogClockRadio.IsChecked.Value)
            {
                type.SetClockType(ClockTypes.ANALOG);
            }
            else
            {
                return;
            }
            foreach (var device in this.deviceList)
            {
                if (device.IsChecked)
                {
                    await this.serverManager.SetClockTypeAsyncTask(type, device.Name);
                }
            }
        }

        private async void SetDateTypeButton_OnClick(object sender, RoutedEventArgs e)
        {
            var type = new DateTypeModel();
            if (this.PersianDateRadio.IsChecked.HasValue && this.PersianDateRadio.IsChecked.Value)
            {
                type.SetDateType(DateTypes.PersianCalendar);
            }
            else if (this.GregorianDateRadio.IsChecked.HasValue && this.GregorianDateRadio.IsChecked.Value)
            {
                type.SetDateType(DateTypes.GregorianCalendar);
            }
            else
            {
                return;
            }
            foreach (var device in this.deviceList)
            {
                if (device.IsChecked)
                {
                    await this.serverManager.SetDateTypeAsyncTask(type, device.Name);
                }
            }
        }

        private async void SyncDateTimeButton_OnClick(object sender, RoutedEventArgs e)
        {
            var now = DateTime.Now;
            try
            {
                var epochTime = StaticHelpers.GetEpochTime(now, TimeZoneInfo.Local);
                var dateData = new DateModel(epochTime);
                var timeZoneInfo = TimeZoneInfo.Local;
                foreach (var device in this.deviceList)
                {
                    if (device.IsChecked)
                    {
                        await this.serverManager.SetTimeZoneAsyncTask(timeZoneInfo, device.Name);
                        await this.serverManager.SetDateTimeAsyncTask(dateData, device.Name);
                    }
                }
            }
            catch (Exception ex)
            {
                Trace.WriteLine(TraceLevel.Error, ex.Message);
            }
        }
    }
}
