﻿// --------------------------------------------------------------------------------------------------------------------
// <copyright file="DeviceLogWindow.xaml.cs" company="Palizafzar">
//   Author : Mohammad Habibzadeh
// </copyright>
// <summary>
//   Device Error Log Window
// </summary>
// --------------------------------------------------------------------------------------------------------------------

namespace PalizTiara.Server
{
    using System;
    using System.Collections.ObjectModel;
    using System.Windows;

    using PalizTiara.Api;
    using PalizTiara.Api.CallBacks;
    using PalizTiara.Api.Helpers;
    using PalizTiara.Api.Models;
    using PalizTiara.Server.ViewModel;

    /// <summary>
    /// Device Error Log Window
    /// </summary>
    public partial class DeviceLogWindow : Window
    {
        private readonly TiaraServerManager serverManager;

        private readonly ObservableCollection<DeviceListBoxModel> deviceList =
            new ObservableCollection<DeviceListBoxModel>();

        private readonly ObservableCollection<DeviceLogDataModel> deviceLogs =
            new ObservableCollection<DeviceLogDataModel>();

        public DeviceLogWindow(TiaraServerManager server)
        {
            this.InitializeComponent();
            this.serverManager = server;
            this.DeviceListBox.ItemsSource = this.deviceList;
            this.LogDataGrid.DataContext = this.deviceLogs;
            this.serverManager.DeviceLogEvent += this.ServerManagerOnDeviceLogEvent;
        }

        protected override void OnClosed(EventArgs e)
        {
            this.serverManager.DeviceLogEvent -= this.ServerManagerOnDeviceLogEvent;
            base.OnClosed(e);
        }

        private void ServerManagerOnDeviceLogEvent(object sender, DeviceLogEventArgs args)
        {
            if (args?.DeviceLogModel?.Logs?.Length > 0)
            {
                this.LogDataGrid.Dispatcher.Invoke(
                    () =>
                        {
                            foreach (var log in args.DeviceLogModel.Logs)
                            {
                                this.deviceLogs.Add(log);
                            }
                        });
            }
        }

        private async void DeviceListBox_OnLoaded(object sender, RoutedEventArgs e)
        {
            var devices = await this.serverManager.GetStatusAsyncTask();
            foreach (var device in devices)
            {
                this.deviceList.Add(new DeviceListBoxModel(device.TerminalName, false));
            }
        }

        private void ClearGridButton_OnClick(object sender, RoutedEventArgs e)
        {
            this.LogDataGrid.Dispatcher.Invoke(
                () =>
                    {
                        this.deviceLogs.Clear();
                    });
        }

        private async void GetLogsButton_OnClick(object sender, RoutedEventArgs e)
        {
            var request = new DeviceLogRequestModel();
            var fromDate = this.FromDatePicker.SelectedDate;
            var toDate = this.ToDatePicker.SelectedDate;
            var pageNumber = 0;
            try
            {
                pageNumber = int.Parse(this.PageTextBox.Text);
            }
            catch (Exception)
            {
                // ignored
            }
            toDate = toDate?.AddHours(23).AddMinutes(59).AddSeconds(59);
            if (fromDate != null)
            {
                request.StartDate = StaticHelpers.GetEpochTime((DateTime)fromDate);
            }
            if (toDate != null)
            {
                request.EndDate = StaticHelpers.GetEpochTime((DateTime)toDate);
            }
            request.Page = pageNumber;
            foreach (var device in this.deviceList)
            {
                if (device.IsChecked)
                {
                    await this.serverManager.GetDeviceLogAsyncTask(device.Name, request);
                }
            }
        }
    }
}
