﻿// --------------------------------------------------------------------------------------------------------------------
// <copyright file="DeviceSerialWindow.xaml.cs" company="Palizafzar">
//   Author : Mohammad Habibzadeh
// </copyright>
// <summary>
//   Interaction logic for DeviceSerialWindow.xaml
// </summary>
// --------------------------------------------------------------------------------------------------------------------

namespace PalizTiara.Server
{
    using System;
    using System.Collections.ObjectModel;
    using System.Windows;
    using System.Windows.Media;

    using PalizTiara.Api;
    using PalizTiara.Api.CallBacks;
    using PalizTiara.Api.Models;
    using PalizTiara.Protocol.Utility;
    using PalizTiara.Server.ViewModel;

    /// <summary>
    /// Interaction logic for DeviceSerialWindow.xaml
    /// </summary>
    public partial class DeviceSerialWindow
    {
        private readonly TiaraServerManager serverManager;
        private readonly ObservableCollection<DeviceSerialListBoxModel> deviceList = new ObservableCollection<DeviceSerialListBoxModel>();

        public DeviceSerialWindow(TiaraServerManager serverManager)
        {
            this.InitializeComponent();
            this.serverManager = serverManager;
            this.DeviceListBox.ItemsSource = this.deviceList;
            this.serverManager.GetSerialEvent += this.ServerManagerOnGetSerialEvent;
            this.serverManager.SetSerialEvent += this.ServerManagerOnSetSerialEvent;
        }

        protected override void OnClosed(EventArgs e)
        {
            this.serverManager.GetSerialEvent -= this.ServerManagerOnGetSerialEvent;
            this.serverManager.SetSerialEvent -= this.ServerManagerOnSetSerialEvent;
            base.OnClosed(e);
        }

        private void ServerManagerOnSetSerialEvent(object sender, SetActionEventArgs args)
        {
            this.MessageTextBlock.Dispatcher.Invoke(
                () =>
                    {
                        if (args.Result)
                        {
                            this.MessageTextBlock.Foreground = Brushes.Green;
                            this.MessageTextBlock.Text = $"Set Serial Done For {args.TerminalName}";
                        }
                        else
                        {
                            this.MessageTextBlock.Foreground = Brushes.Red;
                            this.MessageTextBlock.Text = $"Set Serial Failed For {args.TerminalName}";
                        }
                    });
        }

        private void ServerManagerOnGetSerialEvent(object sender, DeviceSerialEventArgs args)
        {
            var device = (DeviceSender)sender;
            if (!string.IsNullOrEmpty(args?.DeviceSerialModel?.Serial))
            {
                foreach (var d in this.deviceList)
                {
                    if (d.Name.Equals(device.TerminalName))
                    {
                        d.Serial = args.DeviceSerialModel.Serial;
                        Trace.WriteLine(TraceLevel.Error, $"Device Serial : {d.Serial}");
                    }
                }
            }            
        }

        private async void DeviceListBox_OnLoaded(object sender, RoutedEventArgs e)
        {
            var devices = await this.serverManager.GetStatusAsyncTask();
            foreach (var device in devices)
            {
                this.deviceList.Add(new DeviceSerialListBoxModel(device.TerminalName, "None", false));
            }
        }

        private async void GetSerialButton_OnClick(object sender, RoutedEventArgs e)
        {
            foreach (var device in this.deviceList)
            {
                if (device.IsChecked)
                {
                    await this.serverManager.GetDeviceSerialAsyncTask(device.Name);
                }
            }
        }

        private async void SendButton_OnClick(object sender, RoutedEventArgs e)
        {
            if (string.IsNullOrEmpty(this.SerialTexBox.Text))
            {
                this.MessageTextBlock.Foreground = Brushes.Red;
                this.MessageTextBlock.Text = "Please Enter Serial Number";
                return;
            }
            this.MessageTextBlock.Foreground = Brushes.Black;
            this.MessageTextBlock.Text = "Sending...";
            foreach (var device in this.deviceList)
            {
                if (device.IsChecked)
                {
                    await this.serverManager.SetDeviceSerialAsyncTask(device.Name, new DeviceSerialModel(this.SerialTexBox.Text));
                }
            }
            this.MessageTextBlock.Foreground = Brushes.Green;
            this.MessageTextBlock.Text = "Sent!";
        }
    }
}
