﻿// --------------------------------------------------------------------------------------------------------------------
// <copyright file="DeviceSettingsWindow.xaml.cs" company="Palizafzar">
//   Author: Mohammad Habibzadeh
// </copyright>
// <summary>
//   Interaction logic for DeviceSettingsWindow.xaml
// </summary>
// --------------------------------------------------------------------------------------------------------------------

namespace PalizTiara.Server
{
    using System;
    using System.Collections.ObjectModel;
    using System.Windows;

    using Newtonsoft.Json;

    using PalizTiara.Api;
    using PalizTiara.Api.CallBacks;
    using PalizTiara.Api.Models;
    using PalizTiara.Server.ViewModel;

    /// <summary>
    /// Interaction logic for DeviceSettingsWindow.xaml
    /// </summary>
    public partial class DeviceSettingsWindow
    {
        private readonly TiaraServerManager serverManager;

        private readonly ObservableCollection<DeviceListBoxModel> deviceList = new ObservableCollection<DeviceListBoxModel>();

        public DeviceSettingsWindow(TiaraServerManager serverManager)
        {
            this.InitializeComponent();
            this.serverManager = serverManager;
            this.DeviceListBox.ItemsSource = this.deviceList;
            this.serverManager.TiaraSettingsEvent += this.ServerManagerOnTiaraSettingsEvent;
            this.serverManager.SetTiaraSettingsEvent += this.ServerManagerOnSetTiaraSettingsEvent;
            this.serverManager.RebootEvent += this.ServerManagerOnRebootEvent;
        }

        protected override void OnClosed(EventArgs e)
        {
            this.serverManager.TiaraSettingsEvent -= this.ServerManagerOnTiaraSettingsEvent;
            this.serverManager.SetTiaraSettingsEvent -= this.ServerManagerOnSetTiaraSettingsEvent;
            this.serverManager.RebootEvent -= this.ServerManagerOnRebootEvent;
            base.OnClosed(e);
        }

        private void ServerManagerOnRebootEvent(object sender, SetActionEventArgs args)
        {
            this.SettingsTextBlock.Dispatcher.Invoke(
                () =>
                    {
                        this.SettingsTextBlock.Text +=
                            $"\r\nTerminal {args.TerminalName} Reboot Result: {args.Result}\r\n";
                    });
        }

        private async void ServerManagerOnSetTiaraSettingsEvent(object sender, SetActionEventArgs args)
        {
            this.SettingsTextBlock.Dispatcher.Invoke(
                () =>
                {
                    this.SettingsTextBlock.Text +=
                        $"\r\nTerminal {args.TerminalName} Setting Update Result: {args.Result}\r\n";
                });
            if (args.Result)
            {
                await this.serverManager.RebootTiaraAsyncTask(args.TerminalName);
            }
        }

        private void ServerManagerOnTiaraSettingsEvent(object sender, TiaraSettingsEventArgs args)
        {
            var device = (DeviceSender)sender;
            if (args.Result)
            {
                try
                {
                    var jsonSettings = JsonConvert.SerializeObject(args.TiaraSettings);
                    this.SettingsTextBlock.Dispatcher.Invoke(
                        () =>
                            {
                                this.SettingsTextBlock.Text += $"{device.TerminalName} : {jsonSettings}\r\n";
                            });
                    var setting = args.TiaraSettings;
                    setting.ApprovalSetting.FingerprintOnTheCard = false;
                    this.serverManager.SetTiaraSettingsAsyncTask(device.TerminalName, setting);
                }
                catch (Exception e)
                {
                    this.SettingsTextBlock.Dispatcher.Invoke(
                        () =>
                            {
                                this.SettingsTextBlock.Text += $"{device.TerminalName} : Exception: {e.Message}\r\n";
                            });
                }
            }
        }

        private async void DeviceListBox_OnLoaded(object sender, RoutedEventArgs e)
        {
            var devices = await this.serverManager.GetStatusAsyncTask();
            foreach (var device in devices)
            {
                this.deviceList.Add(new DeviceListBoxModel(device.TerminalName, false));
            }
        }

        private async void GetSettingsButton_OnClick(object sender, RoutedEventArgs e)
        {
            foreach (var device in this.deviceList)
            {
                if (device.IsChecked)
                {
                    await this.serverManager.GetTiaraSettingsAsyncTask(device.Name);
                }
            }
        }
    }
}
