﻿// --------------------------------------------------------------------------------------------------------------------
// <copyright file="FailLogWindow.xaml.cs" company="Palizafzar">
//   Author : Mohammad Habibzadeh
// </copyright>
// <summary>
//   Interaction logic for FailLogWindow.xaml
// </summary>
// --------------------------------------------------------------------------------------------------------------------

namespace PalizTiara.Server
{
    using System;
    using System.Collections.ObjectModel;
    using System.Windows;

    using PalizTiara.Api;
    using PalizTiara.Api.CallBacks;
    using PalizTiara.Api.Helpers;
    using PalizTiara.Api.Models;
    using PalizTiara.Server.ViewModel;

    /// <summary>
    /// Interaction logic for FailLogWindow.xaml
    /// </summary>
    public partial class FailLogWindow
    {
        private readonly TiaraServerManager serverManager;
        private readonly ObservableCollection<DeviceListBoxModel> deviceList = new ObservableCollection<DeviceListBoxModel>();
        private readonly ObservableCollection<TrafficLogDataModel> trafficLogs = new ObservableCollection<TrafficLogDataModel>();

        public FailLogWindow(TiaraServerManager serverManager)
        {
            this.InitializeComponent();
            this.serverManager = serverManager;
            this.DeviceListBox.ItemsSource = this.deviceList;
            this.LogDataGrid.DataContext = this.trafficLogs;
            this.serverManager.FailLogEvent += this.ServerManagerOnFailLogEvent;
        }

        protected override void OnClosed(EventArgs e)
        {
            this.serverManager.FailLogEvent -= this.ServerManagerOnFailLogEvent;
            base.OnClosed(e);
        }

        private void ServerManagerOnFailLogEvent(object sender, LogRequestEventArgs args)
        {
            if (args?.TrafficLogModel?.Logs?.Length > 0)
            {
                this.LogDataGrid.Dispatcher.Invoke(() =>
                {
                    foreach (var log in args.TrafficLogModel.Logs)
                    {
                        this.trafficLogs.Add(log);
                    }
                });
            }
        }

        private async void DeviceListBox_OnLoaded(object sender, RoutedEventArgs e)
        {
            var devices = await this.serverManager.GetStatusAsyncTask();
            foreach (var device in devices)
            {
                this.deviceList.Add(new DeviceListBoxModel(device.TerminalName, false));
            }
        }

        private async void GetLogsButton_OnClick(object sender, RoutedEventArgs e)
        {
            var request = new LogRequestModel();
            var fromDate = this.FromDatePicker.SelectedDate;
            var toDate = this.ToDatePicker.SelectedDate;
            long userId = 0;
            try
            {
                userId = long.Parse(this.UserIdTextBox.Text);
            }
            catch (Exception)
            {
            }
            if (fromDate != null)
            {
                request.StartDate = StaticHelpers.GetEpochTime((DateTime)fromDate);
            }
            if (toDate != null)
            {
                request.EndDate = StaticHelpers.GetEpochTime((DateTime)toDate);
            }
            if (userId > 0)
            {
                request.UserId = userId;
            }
            foreach (var device in this.deviceList)
            {
                if (device.IsChecked)
                {
                    await this.serverManager.GetFailLogAsyncTask(device.Name, request);
                }
            }
        }

        private void ClearGridButton_OnClick(object sender, RoutedEventArgs e)
        {
            this.LogDataGrid.Dispatcher.Invoke(() =>
            {
                this.trafficLogs.Clear();
            });
        }
    }
}
