﻿// --------------------------------------------------------------------------------------------------------------------
// <copyright file="LiveTrafficWindow.xaml.cs" company="Palizafzar">
//   Author : Mohammad Habibzadeh
// </copyright>
// <summary>
//   Tiara live traffic test
// </summary>
// --------------------------------------------------------------------------------------------------------------------

namespace PalizTiara.Server
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Windows;
    using System.Windows.Media.Imaging;
    using Api;
    using Api.CallBacks;
    using Api.Helpers;
    using Api.Models;

    /// <summary>
    /// The live traffic window.
    /// </summary>
    public partial class LiveTrafficWindow
    {
        private readonly TiaraServerManager serverManager;
        private readonly ObservableCollection<LiveTrafficModel> liveTraffics;
        private List<DeviceStatusModel> devices;

        /// <summary>
        /// Initializes a new instance of the <see cref="LiveTrafficWindow"/> class.
        /// </summary>
        /// <param name="serverManager">
        /// The server manager.
        /// </param>
        public LiveTrafficWindow(TiaraServerManager serverManager)
        {
            this.InitializeComponent();
            this.liveTraffics = new ObservableCollection<LiveTrafficModel>();
            this.LiveTrafficDataGrid.DataContext = this.liveTraffics;
            this.serverManager = serverManager;
        }

        protected override async void OnClosed(EventArgs e)
        {
            base.OnClosed(e);
            this.serverManager.LiveTrafficLogEvent -= this.ServerManagerOnLiveTrafficLogEvent;
            foreach (var device in this.devices)
            {
                await this.serverManager.StopLiveTrafficLogAsyncTask(device.TerminalName);
            }
        }

        private async void LiveTrafficGrid_OnLoaded(object sender, RoutedEventArgs e)
        {
            this.serverManager.LiveTrafficLogEvent += this.ServerManagerOnLiveTrafficLogEvent;
            this.devices = await this.serverManager.GetStatusAsyncTask();
            foreach (var device in this.devices)
            {
                if (device.IsOnline)
                {
                    await this.serverManager.GetLiveTrafficLogAsyncTask(device.TerminalName);
                }
            }
        }

        private void ServerManagerOnLiveTrafficLogEvent(object sender, LiveTrafficEventArgs args)
        {
            var device = (DeviceSender)sender;
            if (sender == null || args?.LiveTraffic == null)
            {
                return;
            }
            var log = args.LiveTraffic;
            this.LiveTrafficDataGrid.Dispatcher.Invoke(() =>
            {
                BitmapImage bitmapImage = StaticHelpers.ToImage(log.Image);
                if (bitmapImage != null)
                {
                    this.TestImage.Source = bitmapImage;
                }
                this.liveTraffics.Add(new LiveTrafficModel(device.TerminalName, log.Id, log.UserId, log.Valid, StaticHelpers.GetDateTimeFromEpoch(log.Time), StaticHelpers.GetDeviceNames(log.Device), StaticHelpers.GetTrafficName(log.TrafficType)));                
            });
        }

        private void ClearButton_OnClick(object sender, RoutedEventArgs e)
        {
            this.liveTraffics.Clear();
            this.TestImage.Source = null;
        }

        private class LiveTrafficModel
        {
            public LiveTrafficModel()
            {
            }

            public LiveTrafficModel(string terminalName, long id, long userId, bool valid, DateTime time, string device, string trafficType)
            {
                this.TerminalName = terminalName;
                this.Id = id;
                this.UserId = userId;
                this.Valid = valid;
                this.Time = time;
                this.Device = device;
                this.TrafficType = trafficType;
            }

            public string TerminalName { get; set; }

            public long Id { get; set; }

            public long UserId { get; set; }

            public bool Valid { get; set; }

            public DateTime Time { get; set; }

            public string Device { get; set; }

            public string TrafficType { get; set; }
        }
    }
}
