﻿// --------------------------------------------------------------------------------------------------------------------
// <copyright file="MassUserInfoCopyWindow.xaml.cs" company="Palizafzar">
//   Author: Mohammad Habibzadeh
// </copyright>
// <summary>
//   Interaction logic for MassUserInfoCopyWindow.xaml
// </summary>
// --------------------------------------------------------------------------------------------------------------------

namespace PalizTiara.Server
{
    using System;
    using System.Collections.ObjectModel;
    using System.Threading.Tasks;
    using System.Windows;

    using PalizTiara.Api;
    using PalizTiara.Api.CallBacks;
    using PalizTiara.Api.Models;
    using PalizTiara.Server.ViewModel;

    /// <summary>
    /// Interaction logic for MassUserInfoCopyWindow.xaml
    /// </summary>
    public partial class MassUserInfoCopyWindow : Window
    {
        private const int BatchSize = 100;
        private readonly TiaraServerManager serverManager;
        private readonly ObservableCollection<DeviceListBoxModel> sourceDeviceList = new ObservableCollection<DeviceListBoxModel>();
        private readonly ObservableCollection<DeviceListBoxModel> destDeviceList = new ObservableCollection<DeviceListBoxModel>();

        public MassUserInfoCopyWindow(TiaraServerManager serverManager)
        {
            this.InitializeComponent();
            this.serverManager = serverManager;
            this.SourceListBox.ItemsSource = this.sourceDeviceList;
            this.DestListBox.ItemsSource = this.destDeviceList;
        }

        protected override void OnClosed(EventArgs e)
        {
            this.serverManager.DeleteAllUsersEvent -= this.ServerManagerOnDeleteAllUsersEvent;
            this.serverManager.UserListEvent -= this.ServerManagerOnUserListEvent;
            this.serverManager.MassUserInfoEvent -= this.ServerManagerOnMassUserInfoEvent;
            this.serverManager.MassUserInsertEvent -= this.ServerManagerOnMassUserInsertEvent;
            base.OnClosed(e);
        }

        private async void MassUserInfoCopyWindow_OnLoaded(object sender, RoutedEventArgs e)
        {
            var devices = await this.serverManager.GetStatusAsyncTask();
            foreach (var device in devices)
            {
                this.sourceDeviceList.Add(new DeviceListBoxModel(device.TerminalName, false));
                this.destDeviceList.Add(new DeviceListBoxModel(device.TerminalName, false));
            }
            this.serverManager.DeleteAllUsersEvent += this.ServerManagerOnDeleteAllUsersEvent;
            this.serverManager.UserListEvent += this.ServerManagerOnUserListEvent;
            this.serverManager.MassUserInfoEvent += this.ServerManagerOnMassUserInfoEvent;
            this.serverManager.MassUserInsertEvent += this.ServerManagerOnMassUserInsertEvent;
        }

        private void ServerManagerOnMassUserInsertEvent(object sender, MassUserActionEventArgs args)
        {
            if (args == null || !args.Result)
            {
                this.Log($"{((DeviceSender)sender).TerminalName} failed to add users.");          
            }
            else
            {
                this.Log($"Users added to {((DeviceSender)sender).TerminalName}.");
            }
        }

        private async void ServerManagerOnMassUserInfoEvent(object sender, MassUserInfoEventArgs args)
        {
            if (args == null || !args.Result || args.Users?.UserInfoModels == null || args.Users.UserInfoModels.Length == 0)
            {
                this.Log($"{((DeviceSender)sender).TerminalName} failed to send user informations.");
                return;
            }
            foreach (var device in this.destDeviceList)
            {
                if (device.IsChecked)
                {
                    this.Log($"Sending users to {device.Name}");
                    this.Log($"First user id: {args.Users.UserInfoModels[0].Id}");
                    await this.serverManager.MassUserInsertTask(device.Name, args.Users);
                }
            }
        }

        private async void ServerManagerOnUserListEvent(object sender, UserListEventArgs args)
        {
            var deviceSender = (DeviceSender)sender;
            if (args?.UserListModel?.UserIds != null && args.UserListModel.UserIds.Length > 0)
            {
                this.Log($"{DateTime.Now}:User List Received From {deviceSender.TerminalName}");
                var total = args.UserListModel.UserIds.Length;
                var steps = 0;
                var remained = 0;
                if (total < BatchSize)
                {
                    steps = 1;
                    remained = total;
                }
                else
                {
                    steps = total / BatchSize;
                    remained = total % BatchSize;
                    if (remained > 0)
                    {
                        steps++;
                    }
                }

                for (var i = 0; i < steps; i++)
                {
                    long from;
                    long to;
                    if (i == (steps - 1))
                    {
                        from = args.UserListModel.UserIds[i * BatchSize];
                        to = args.UserListModel.UserIds[((i * BatchSize) + remained) - 1];
                        await this.serverManager.GetMassUserInfoTask(
                            deviceSender.TerminalName,
                            new MassUserIdModel(from, to));
                    }
                    else
                    {
                        from = args.UserListModel.UserIds[i * BatchSize];
                        to = args.UserListModel.UserIds[((i * BatchSize) + BatchSize) - 1];
                        await this.serverManager.GetMassUserInfoTask(deviceSender.TerminalName, new MassUserIdModel(from, to));
                    }
                    this.Log($"Device: {deviceSender.TerminalName}, From: {from}, To: {to}");
                }
            }
            else if (args != null && !args.Result)
            {
                this.Log($"{DateTime.Now}:Get User List Failed From {deviceSender.TerminalName}");
            }
        }

        private void ServerManagerOnDeleteAllUsersEvent(object sender, SetActionEventArgs args)
        {
            this.Log(args.Result
                ? $"Terminal {args.TerminalName} Cleared!"
                : $"Delete All User Failed For Terminal {args.TerminalName}");
        }

        private async void StartCopyButton_OnClick(object sender, RoutedEventArgs e)
        {
            await this.GetUsers();
        }

        private Task GetUsers()
        {
            return Task.Run(async () =>
            {
                foreach (var device in this.sourceDeviceList)
                {
                    if (device.IsChecked)
                    {
                        this.Log($"Get User List From Terminal {device.Name}");
                        await this.serverManager.GetUserListAsyncTask(device.Name);
                    }
                }
            });
        }

        private async void DeleteAllUsers_OnClick(object sender, RoutedEventArgs e)
        {
            foreach (var device in this.destDeviceList)
            {
                if (device.IsChecked)
                {
                    this.Log($"Delete All Information From Terminal {device.Name}");
                    await this.serverManager.DeleteAllUsersAsyncTask(device.Name);
                }
            }
        }

        private void Log(string log)
        {
            this.LogTextBlock.Dispatcher.Invoke(() =>
            {
                this.LogTextBlock.Text += $"{log}{Environment.NewLine}";
            });
        }
    }
}
