﻿// --------------------------------------------------------------------------------------------------------------------
// <copyright file="RemoteHardwareWindow.xaml.cs" company="Palizafzar">
//   Author : Mohammad Habibzadeh
// </copyright>
// <summary>
//   Interaction logic for RemoteHardwareWindow.xaml
// </summary>
// --------------------------------------------------------------------------------------------------------------------

namespace PalizTiara.Server
{
    using System;
    using System.Collections.ObjectModel;
    using System.Windows;

    using PalizTiara.Api;
    using PalizTiara.Api.CallBacks;
    using PalizTiara.Api.Models;
    using PalizTiara.Server.ViewModel;

    /// <summary>
    /// Interaction logic for RemoteHardwareWindow.xaml
    /// </summary>
    public partial class RemoteHardwareWindow
    {
        private readonly TiaraServerManager serverManager;
        private readonly ObservableCollection<DeviceListBoxModel> deviceList = new ObservableCollection<DeviceListBoxModel>();

        public RemoteHardwareWindow(TiaraServerManager serverManager)
        {
            this.InitializeComponent();
            this.serverManager = serverManager;
            this.DeviceListBox.ItemsSource = this.deviceList;
            this.serverManager.RemoteHardwareCardEvent += this.ServerManagerOnRemoteHardwareCardEvent;
            this.serverManager.RemoteHardwareFingerprintEvent += this.ServerManagerOnRemoteHardwareFingerprintEvent;
        }

        protected override void OnClosed(EventArgs e)
        {
            this.serverManager.RemoteHardwareCardEvent -= this.ServerManagerOnRemoteHardwareCardEvent;
            this.serverManager.RemoteHardwareFingerprintEvent -= this.ServerManagerOnRemoteHardwareFingerprintEvent;
            base.OnClosed(e);
        }

        private void ServerManagerOnRemoteHardwareFingerprintEvent(object sender, RemoteHardwareFingerprintEventArgs args)
        {
            if (sender == null || args == null)
            {
                return;
            }
            var device = (DeviceSender)sender;
            this.FingerprintTextBlock.Dispatcher.Invoke(
                () =>
                    {
                this.FingerprintTextBlock.Text +=
                    $"Terminal: {device.TerminalName}, Status: {args.Result}, Fingerprint Length: {args.Fingerprint?.Fingerprint?.Length}, Fingerprint Image Legnth: {args.Fingerprint?.Image?.Length}\r\n";
            });
        }

        private void ServerManagerOnRemoteHardwareCardEvent(object sender, RemoteHardwareCardEventArgs args)
        {
            if (sender == null || args == null)
            {
                return;
            }
            var device = (DeviceSender) sender;
            this.CardReaderTextBlock.Dispatcher.Invoke(() =>
            {
                this.CardReaderTextBlock.Text +=
                    $"Terminal: {device.TerminalName}, Status: {args.Result}, Card: {args.Card?.CardNumber}\r\n";
            });
        }

        private async void DeviceListBox_OnLoaded(object sender, RoutedEventArgs e)
        {
            var devices = await this.serverManager.GetStatusAsyncTask();
            foreach (var device in devices)
            {
                this.deviceList.Add(new DeviceListBoxModel(device.TerminalName, false));
            }
        }

        private async void CardReaderButton_OnClick(object sender, RoutedEventArgs e)
        {
            foreach (var device in this.deviceList)
            {
                if (device.IsChecked)
                {
                    await this.serverManager.RemoteCardRequestAsyncTask(device.Name, 0);
                }
            }
        }

        private async void FingerprintButton_OnClick(object sender, RoutedEventArgs e)
        {
            foreach (var device in this.deviceList)
            {
                if (device.IsChecked)
                {
                    await this.serverManager.RemoteFingerprintRequestAsyncTask(device.Name, 0);
                }
            }
        }
    }
}
