﻿// --------------------------------------------------------------------------------------------------------------------
// <copyright file="UserInfoCopyWindow.xaml.cs" company="Palizafzar">
//   Author : Mohammad Habibzadeh
// </copyright>
// <summary>
//   
// </summary>
// --------------------------------------------------------------------------------------------------------------------

namespace PalizTiara.Server
{
    using System;
    using System.Collections.ObjectModel;
    using System.Threading.Tasks;
    using System.Windows;
    using Api;
    using Api.CallBacks;
    using Api.Models;
    using ViewModel;

    /// <summary>
    /// User Information Window
    /// </summary>
    public partial class UserInfoCopyWindow
    {
        private readonly TiaraServerManager serverManager;
        private readonly ObservableCollection<DeviceListBoxModel> sourceDeviceList = new ObservableCollection<DeviceListBoxModel>();
        private readonly ObservableCollection<DeviceListBoxModel> destDeviceList = new ObservableCollection<DeviceListBoxModel>();

        public UserInfoCopyWindow(TiaraServerManager serverManager)
        {
            this.InitializeComponent();
            this.serverManager = serverManager;
            this.SourceListBox.ItemsSource = this.sourceDeviceList;
            this.DestListBox.ItemsSource = this.destDeviceList;
            this.serverManager.DeleteAllUsersEvent += this.ServerManagerOnDeleteAllUsersEvent;
            this.serverManager.UserListEvent += this.ServerManagerOnUserListEvent;
            this.serverManager.AddUserEvent += this.ServerManagerOnAddUserEvent;
            this.serverManager.UserInfoEvent += this.ServerManagerOnUserInfoEvent;
        }

        protected override void OnClosed(EventArgs e)
        {
            this.serverManager.DeleteAllUsersEvent -= this.ServerManagerOnDeleteAllUsersEvent;
            this.serverManager.UserListEvent -= this.ServerManagerOnUserListEvent;
            this.serverManager.AddUserEvent -= this.ServerManagerOnAddUserEvent;
            this.serverManager.UserInfoEvent -= this.ServerManagerOnUserInfoEvent;
            base.OnClosed(e);
        }

        private async void ServerManagerOnUserInfoEvent(object sender, UserInfoEventArgs args)
        {
            var deviceSender = (DeviceSender)sender;
            if (args?.UserInfoModel != null && args.UserInfoModel.Id > 0)
            {
                if (args.Result)
                {
                    this.Log($"{DateTime.Now}:User Information Received For Id {args.UserInfoModel.Id} From {deviceSender.TerminalName}");
                    foreach (var device in this.destDeviceList)
                    {
                        if (device.IsChecked)
                        {
                            await this.serverManager.AddUserAsyncTask(device.Name, args.UserInfoModel);
                        }
                    }
                }
                else
                {
                    this.Log($"{DateTime.Now}:Get User Information Failed For Id {args.UserInfoModel.Id} From {deviceSender.TerminalName}");
                }
            }
            else
            {
                this.Log($"User Information Failed For Unknown User From {deviceSender.TerminalName}");
            }
        }

        private void ServerManagerOnAddUserEvent(object sender, UserActionEventArgs args)
        {
            this.Log(args.Result
                ? $"{DateTime.Now}:User {args.UserId.UserId} Added To Terminal {args.TerminalName}"
                : $"{DateTime.Now}:User {args.UserId.UserId} Add Failed On Terminal {args.TerminalName}");
        }

        private async void ServerManagerOnUserListEvent(object sender, UserListEventArgs args)
        {
            var deviceSender = (DeviceSender)sender;
            if (args?.UserListModel?.UserIds != null && args.UserListModel.UserIds.Length > 0)
            {
                this.Log($"{DateTime.Now}:User List Received From {deviceSender.TerminalName}");
                foreach (var userId in args.UserListModel.UserIds)
                {
                    this.Log($"{DateTime.Now}:Get User {userId} Info Terminal {deviceSender.TerminalName}");
                    await this.serverManager.GetUserInfoAsyncTask(new UserIdModel(userId), deviceSender.TerminalName);
                   
                }
            }
            else if (args != null && !args.Result)
            {
                this.Log($"{DateTime.Now}:Get User List Failed From {deviceSender.TerminalName}");
            }                                      
        }

        private void ServerManagerOnDeleteAllUsersEvent(object sender, SetActionEventArgs args)
        {
            this.Log(args.Result
                ? $"Terminal {args.TerminalName} Cleared!"
                : $"Delete All User Failed For Terminal {args.TerminalName}");
        }

        private async void StartCopyButton_OnClick(object sender, RoutedEventArgs e)
        {
            await this.GetUsers();
        }

        private Task GetUsers()
        {
            return Task.Run(async () =>
            {
                foreach (var device in this.sourceDeviceList)
                {
                    if (device.IsChecked)
                    {
                        this.Log($"Get User List From Terminal {device.Name}");
                        await this.serverManager.GetUserListAsyncTask(device.Name);
                    }
                }
            });
        }

        private void Log(string log)
        {
            this.LogTextBlock.Dispatcher.Invoke(() =>
            {
                this.LogTextBlock.Text += $"{log}{Environment.NewLine}";
            });
        }

        private async void UserInfoCopyWindow_OnLoaded(object sender, RoutedEventArgs e)
        {
            var devices = await this.serverManager.GetStatusAsyncTask();
            foreach (var device in devices)
            {
                this.sourceDeviceList.Add(new DeviceListBoxModel(device.TerminalName, false));
                this.destDeviceList.Add(new DeviceListBoxModel(device.TerminalName, false));
            }
        }

        private async void DeleteAllUsers_OnClick(object sender, RoutedEventArgs e)
        {
            foreach (var device in this.destDeviceList)
            {
                if (device.IsChecked)
                {
                    this.Log($"Delete All Information From Terminal {device.Name}");
                    await this.serverManager.DeleteAllUsersAsyncTask(device.Name);
                }
            }
        }
    }
}
