﻿// --------------------------------------------------------------------------------------------------------------------
// <copyright file="UserInfoWindow.xaml.cs" company="Palizafzar">
//   Author : Mohammad Habibzadeh
// </copyright>
// <summary>
//   User Information Window
// </summary>
// --------------------------------------------------------------------------------------------------------------------

namespace PalizTiara.Server
{
    using System;
    using System.Collections.ObjectModel;
    using System.Windows;
    using System.Windows.Input;

    using PalizTiara.Api;
    using PalizTiara.Api.CallBacks;
    using PalizTiara.Api.Models;
    using PalizTiara.Server.ViewModel;

    /// <summary>
    /// User Information Window
    /// </summary>
    public partial class UserInfoWindow
    {
        private readonly TiaraServerManager serverManager;
        private readonly ObservableCollection<DeviceListBoxModel> deviceList = new ObservableCollection<DeviceListBoxModel>();
        private readonly ObservableCollection<UserDeviceIdModel> userIds = new ObservableCollection<UserDeviceIdModel>();
        private readonly ObservableCollection<UserInfoModel> userInformations = new ObservableCollection<UserInfoModel>();

        public UserInfoWindow(TiaraServerManager serverManager)
        {
            this.InitializeComponent();
            this.serverManager = serverManager;
            this.DeviceListBox.ItemsSource = this.deviceList;
            this.UserIdGrid.ItemsSource = this.userIds;
            this.UserInfoDetailGrid.ItemsSource = this.userInformations;
            this.serverManager.UserInfoEvent += this.ServerManagerOnUserInfoEvent;
            this.serverManager.UserListEvent += this.ServerManagerOnUserListEvent;
        }

        protected override void OnClosed(EventArgs e)
        {
            this.serverManager.UserListEvent -= this.ServerManagerOnUserListEvent;
            this.serverManager.UserInfoEvent -= this.ServerManagerOnUserInfoEvent;
            base.OnClosed(e);
        }

        private async void DeviceListBox_OnLoaded(object sender, RoutedEventArgs e)
        {
            var devices = await this.serverManager.GetStatusAsyncTask();
            foreach (var device in devices)
            {
                this.deviceList.Add(new DeviceListBoxModel(device.TerminalName, false));
            }
        }

        private async void GetUsersButton_OnClick(object sender, RoutedEventArgs e)
        {
            foreach (var device in this.deviceList)
            {
                if (device.IsChecked)
                {
                    await this.serverManager.GetUserListAsyncTask(device.Name);
                }
            }
        }

        private void ServerManagerOnUserListEvent(object sender, UserListEventArgs args)
        {
            var deviceSender = (DeviceSender)sender;
            if (args == null)
            {
                return;
            }
            foreach (var id in args.UserListModel.UserIds)
            {
                this.UserIdGrid.Dispatcher.Invoke(() =>
                {
                    this.userIds.Add(new UserDeviceIdModel(id, deviceSender.TerminalName));
                });
            }
        }

        private async void UserIdDoubleClickListener(object sender, MouseButtonEventArgs e)
        {
            var user = (UserDeviceIdModel)this.UserIdGrid.SelectedValue;
            await this.serverManager.GetUserInfoAsyncTask(new UserIdModel(user.UserId), user.TerminalName);
        }

        private void ServerManagerOnUserInfoEvent(object sender, UserInfoEventArgs args)
        {
            if (args?.UserInfoModel != null)
            {
                this.UserInfoDetailGrid.Dispatcher.Invoke(() =>
                {
                    this.userInformations.Add(args.UserInfoModel);
                });
            }
        }
    }
}
